//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//
/*++


Module Name:

    mon.c

Abstract:

    Power Manager Monitor.
    Run in the background to display power state changes in debugger.

Notes:

Revision History:
	ST - adapted from mon.c - platform builder 4.2

--*/
#include "stdafx.h"
#include <windows.h>
#include <msgqueue.h>
#include <pm.h>

/*  
	Errors related to this on compile may be related to the wrong SDK installation
	Check you are using Embedded Visual C++ 4 and have the appropriate SDKs installed
	see microsoft web page on embedded development for further details
*/

#include "VCPTestDlg.h"

static BOOL bRunning = FALSE;

#define QUEUE_ENTRIES   3
#define MAX_NAMELEN     200
#define QUEUE_SIZE      (QUEUE_ENTRIES * (sizeof(POWER_BROADCAST) + MAX_NAMELEN))

BOOL StopPowerManagement()
{
	bRunning = FALSE;
	return TRUE;
}

//****************************************************************************
UINT PowerManagementThread(LPVOID param)//count up thread
{
	CVCPTestDlg *pMyHndl = (CVCPTestDlg *)param;
    HANDLE hMsgQ = NULL;
    HANDLE hNotifications = NULL;
    DWORD dwErr, dwFlags = 0, dwCount = 0;
    DWORD dwPri = CeGetThreadPriority(GetCurrentThread());
	HWND hWnd = (HWND)param;
    MSGQUEUEOPTIONS msgOptions = {0};   
    UCHAR buf[QUEUE_SIZE];
    DWORD iBytesInQueue;
	
    // create a message queue for Power Manager notifications
    //
    msgOptions.dwSize = sizeof(MSGQUEUEOPTIONS);
    msgOptions.dwFlags = 0;
    msgOptions.dwMaxMessages = QUEUE_ENTRIES;
    msgOptions.cbMaxMessage = sizeof(POWER_BROADCAST) + MAX_NAMELEN;
    msgOptions.bReadAccess = TRUE;

    hMsgQ = CreateMsgQueue(NULL, &msgOptions);
    if (!hMsgQ) {
        dwErr = GetLastError();
        RETAILMSG(1, (TEXT("PMMON!CreateMessageQueue ERROR:%d\n"), dwErr));
        goto _Exit;
    }

    // request Power notifications
    //
    hNotifications = RequestPowerNotifications(hMsgQ,
                                               POWER_NOTIFY_ALL); // Flags
    if (!hNotifications) {
        dwErr = GetLastError();
        RETAILMSG(1, (TEXT("PMMON!RequestPowerNotifications ERROR:%d\n"), dwErr));
        goto _Exit;
    }

	bRunning = TRUE;

    while (bRunning) 
    {
        iBytesInQueue = 0;
        memset(&buf, 0, QUEUE_SIZE);

        // Block on our message queue - can block or not - depending on your preference
		// Blocking will use less power but unable to clean up thread without a TerminateThread call
        // This thread runs when the power manager writes a notification into the queue.
        if ( !ReadMsgQueue(hMsgQ,
                           &buf,
                           QUEUE_SIZE,
                           &iBytesInQueue,
                           INFINITE,    // Timeout
                           //5000,    // Timeout
                           &dwFlags))
        {
            dwErr = GetLastError();
            RETAILMSG(1, (TEXT("PMMON!ReadMsgQueue: ERROR:%d\n"), dwErr));
            ASSERT(0);
        } else if(iBytesInQueue >= sizeof(POWER_BROADCAST)) {
            //
            // process power notifications
            //
            PPOWER_BROADCAST pB = (PPOWER_BROADCAST)&buf;
            
            RETAILMSG(1, (TEXT("PMMON!*** Power Notification @ Tick:%u, Count:%d, Pri:%d***\n"), 
                GetTickCount(), dwCount++, dwPri));
            switch (pB->Message) 
            {
            case PBT_TRANSITION:
                RETAILMSG(1, (TEXT("PMMON!\tPBT_TRANSITION to system power state: '%s'\n"), pB->SystemPowerState));
                switch (POWER_STATE(pB->Flags)) {
                case POWER_STATE_ON:
                    RETAILMSG(1, (TEXT("PMMON!\tPOWER_STATE_ON\n")));
                    break;
                case POWER_STATE_OFF:
					RETAILMSG(1, (TEXT("PMMON!\tPOWER_STATE_OFF\n")));
                    break;
                case POWER_STATE_CRITICAL:
                    RETAILMSG(1, (TEXT("PMMON!\tPOWER_STATE_CRITICAL\n")));
                    break;
                case POWER_STATE_BOOT:
                    RETAILMSG(1, (TEXT("PMMON!\tPOWER_STATE_BOOT\n")));
                    break;
                case POWER_STATE_IDLE:
                    RETAILMSG(1, (TEXT("PMMON!\tPOWER_STATE_IDLE\n")));
                    break;
                case POWER_STATE_SUSPEND:
					pMyHndl->ClosePort();
                    RETAILMSG(1, (TEXT("PMMON!\tPOWER_STATE_SUSPEND\n")));
                    break;
                case POWER_STATE_RESET:
                    RETAILMSG(1, (TEXT("PMMON!\tPOWER_STATE_RESET\n")));
                    break;
                case 0:
                    break;
                default:
                    RETAILMSG(1,(TEXT("PMMON!\tUnknown Power State Flags:0x%x\n"),pB->Flags));
                    ASSERT(0);
                    break;
                }
                break;
                
                case PBT_RESUME:
                    RETAILMSG(1, (TEXT("PMMON!\tPBT_RESUME\n")));
                    break;
                    
                case PBT_POWERSTATUSCHANGE:
                    RETAILMSG(1, (TEXT("PMMON!\tPBT_POWERSTATUSCHANGE\n")));
                    break;
                    
                case PBT_POWERINFOCHANGE:
                    {
                        PPOWER_BROADCAST_POWER_INFO ppbpi = (PPOWER_BROADCAST_POWER_INFO) pB->SystemPowerState;

                        RETAILMSG(1, (TEXT("PMMON!\tPBT_POWERINFOCHANGE\n")));
                        RETAILMSG(1, (TEXT("PMMON!\t\tAC line status %u, battery flag %u, backup flag %u, %d levels\n"),
                            ppbpi->bACLineStatus, ppbpi->bBatteryFlag, ppbpi->bBackupBatteryFlag, ppbpi->dwNumLevels));
                        RETAILMSG(1, (TEXT("PMMON!\t\tbattery life %d, backup life %d\n"),
                            ppbpi->bBatteryLifePercent, ppbpi->bBackupBatteryLifePercent));
                        RETAILMSG(1, (TEXT("PMMON!\t\tlifetime 0x%08x, full lifetime 0x%08x\n"),
                            ppbpi->dwBatteryLifeTime, ppbpi->dwBatteryFullLifeTime));
                        RETAILMSG(1, (TEXT("PMMON!\t\tbackup lifetime 0x%08x, backup full lifetime 0x%08x\n"),
                            ppbpi->dwBackupBatteryLifeTime, ppbpi->dwBackupBatteryFullLifeTime));
                    }
                    break;
                    
                default:
                    RETAILMSG(1, (TEXT("PMMON!\tUnknown Message:%d\n"), pB->Message));
                    ASSERT(0);
                    break;
            }

            RETAILMSG(1, (TEXT("PMMON!\tMessage: 0x%x\n"), pB->Message));
            RETAILMSG(1, (TEXT("PMMON!\tFlags: 0x%x\n"), pB->Flags));
            RETAILMSG(1, (TEXT("PMMON!\tdwLen: %d\n"), pB->Length));
            
            RETAILMSG(1, (TEXT("PMMON!***********************\n")));
        } else {
            RETAILMSG(1, (TEXT("PMMON!\tReceived short message: %d bytes\n"), iBytesInQueue));
            ASSERT(0);
        }
    }

_Exit:
    if (hMsgQ)
        CloseMsgQueue(hMsgQ);

    if (hNotifications && !StopPowerNotifications(hNotifications)) {
        RETAILMSG(1, (TEXT("PMMON!StopNotifications ERROR:%d\n"), GetLastError()));
        ASSERT(0);
    }
    ExitThread(ERROR_SUCCESS);  

    return ERROR_SUCCESS;
}

