// VCPTestDlg.cpp : implementation file
//

/*
	This multithreaded MFC based application is a simple example on how to 
	1. Enumerate FTDI only VCP devices
	2. Open, Close, Read and Write to the device
	3. Simple power management to detect suspend - you must close the device on a suspend detect.

	A serial cable with a suitable loopback connector should be used to get sensible results when writing to port.
	For an example see www.easysync.co.uk for a suitable product
*/

#include "stdafx.h"
#include "VCPTest.h"
#include "VCPTestDlg.h"
#include "PowerMgmt.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define REG_NAME_KEY		L"Key"
#define REG_NAME_NAME		L"Name"
#define REG_NAME_PREFIX		L"Prefix"
#define REG_NAME_FTDI_KEY	L"Drivers\\USB\\FTDI_DEVICE"
#define REG_NAME_ACTIVE		L"Drivers\\Active"

#define POWER_THREAD_WAIT_TIMEOUT	5000
#define READ_THREAD_WAIT_TIMEOUT	5000
#define WRITE_THREAD_WAIT_TIMEOUT	5000

UINT SendThread (LPVOID pArg);
UINT ReadThread (LPVOID pArg);

/////////////////////////////////////////////////////////////////////////////
// CVCPTestDlg dialog

CVCPTestDlg::CVCPTestDlg(CWnd* pParent /*=NULL*/)
	: CDialog(CVCPTestDlg::IDD, pParent)
{
	//{{AFX_DATA_INIT(CVCPTestDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT
	// Note that LoadIcon does not require a subsequent DestroyIcon in Win32
	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);
	m_hPort = NULL;
	fContinue = TRUE;
	sReadBuffer.size = COM_BUF_SIZE;
	sWriteBuffer.size = COM_BUF_SIZE;
}

void CVCPTestDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CVCPTestDlg)
	DDX_Control(pDX, IDC_READ_LIST, m_ReadList);
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CVCPTestDlg, CDialog)
	//{{AFX_MSG_MAP(CVCPTestDlg)
	ON_BN_CLICKED(IDC_OPEN, OnOpen)
	ON_BN_CLICKED(IDC_CLOSE, OnClose)
	ON_BN_CLICKED(IDC_WRITE, OnWrite)
	ON_WM_DESTROY()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CVCPTestDlg message handlers

BOOL CVCPTestDlg::OnInitDialog()
{
	CDialog::OnInitDialog();
	CString str;

	// Set the icon for this dialog.  The framework does this automatically
	//  when the application's main window is not a dialog
	SetIcon(m_hIcon, TRUE);			// Set big icon
	SetIcon(m_hIcon, FALSE);		// Set small icon
	
	CenterWindow(GetDesktopWindow());	// center to the hpc screen

	str.Format(L"%d", COM_BUF_SIZE);
	SetDlgItemText(IDC_WRITE_COUNT, str);
	SetUpComList(IDC_CMB_PORTS);

	pThreadPower = AfxBeginThread(PowerManagementThread, this, THREAD_PRIORITY_NORMAL);
	GetDlgItem(IDC_CLOSE)->EnableWindow(FALSE);
	GetDlgItem(IDC_OPEN)->EnableWindow(TRUE);
	GetDlgItem(IDC_WRITE)->EnableWindow(FALSE);

	return TRUE;  // return TRUE  unless you set the focus to a control
}

//
// Open button press event handler
//
void CVCPTestDlg::OnOpen() 
{
	OpenPort();
}

//
// Close button press event handler
//
void CVCPTestDlg::OnClose() 
{
	ClosePort();
}

//
// Write button press event handler
//
void CVCPTestDlg::OnWrite() 
{
	BOOL bTrans;
	INT iCount;		

	if(m_hPort == NULL) {
		MessageBox(L"Error", L"Port not open", MB_OK);
		return;
	}

	iCount = (INT)GetDlgItemInt(IDC_WRITE_COUNT, &bTrans, FALSE);

	if(bTrans == TRUE) {
		if((iCount >= COM_BUF_SIZE) && (iCount < 0)) 
			iCount = COM_BUF_SIZE-1;
	}
	else {
		iCount = COM_BUF_SIZE-1;	// always write something
	}

	SetDlgItemInt(IDC_WRITE_COUNT, iCount);

	for(int i = 0; i < iCount; i++) {
		sWriteBuffer.cBuf[i] = (char)(i & 0xFF);
	}
	sWriteBuffer.size = iCount;
	SetEvent(m_hEventWrite);
}

BOOL CVCPTestDlg::DestroyWindow() 
{
	ClosePort();
	return CDialog::DestroyWindow();
}

//
//	Name: SendThread
//
//
//	Purpose: Signal from a button press this will send characters to the serial port
//
UINT SendThread (LPVOID pArg) 
{
    DWORD dwBytes, dwGoCode;
	CVCPTestDlg *pMyHndl = (CVCPTestDlg *)pArg;

	//
	// Infinite loop to write to port until close
	//
    while (pMyHndl->fContinue) {

		//
		// Wait for a signal from button press to write
		//
        dwGoCode = WaitForSingleObject (pMyHndl->m_hEventWrite, INFINITE);
        if (dwGoCode == WAIT_OBJECT_0) {

			//
			// Write size worth of data to the port
			//
            WriteFile (
				pMyHndl->m_hPort,
				pMyHndl->sWriteBuffer.cBuf, 
				pMyHndl->sWriteBuffer.size,
                &dwBytes, 
				0
				);
        } 
		else
			return 1;
    }
    return 0;
}

//
//	Name: ReadThread
//
//
//	Purpose: Continually receives characters from the serial port if available
//
UINT ReadThread (LPVOID pArg) 
{
    DWORD dwBytes;
    BYTE szText[TEXTSIZE], *pPtr;
	CVCPTestDlg *pMyHndl = (CVCPTestDlg *)pArg;
	CString str;

	//
	// Infinite loop to read from port until close
	//
    while (pMyHndl->fContinue) {
        pPtr = szText;
		if (pMyHndl->m_hPort == INVALID_HANDLE_VALUE)
			return 1;
		//
		// Read 1 byte - not very efficient this way
		//
		ReadFile(
			pMyHndl->m_hPort, 
			pMyHndl->sReadBuffer.cBuf, 
			1, 
			&dwBytes, 
			0
			);

		if(dwBytes) {
			if(pMyHndl->m_ReadList.GetCount() > COM_BUF_SIZE-1)
				pMyHndl->m_ReadList.ResetContent();
			str.Format(L"0x%02X", (pMyHndl->sReadBuffer.cBuf[0] & 0xFF));
			pMyHndl->m_ReadList.AddString(str);
		}
    }

    return 0;
}

void CVCPTestDlg::OnDestroy() 
{
	//
	// Terminate the Power management thread
	// 
	StopPowerManagement();
	if(WaitForSingleObject(pThreadPower->m_hThread, POWER_THREAD_WAIT_TIMEOUT) == WAIT_TIMEOUT) {
		//
		// Didnt exit nicely so kill it
		//
		TerminateThread(pThreadPower->m_hThread, 0xffffffff);
	}
	CDialog::OnDestroy();	
}


//
//	Name: OpenPort
//
//
//	Purpose:	Open the selected port from the list box
//				Setup port if opened
//				Start the Send and Receive threads
//
BOOL CVCPTestDlg::OpenPort()
{
	CString str;
	COMMTIMEOUTS cto;
	DCB dcbCommPort;

	if(!m_hPort) {
		//
		// Get Selected COM (or other if prefix has changed) port name from list box
		//
		GetDlgItemText(IDC_CMB_PORTS, str);

		//
		// Open the port
		//
		m_hPort = CreateFile (
					str,							// Pointer to the name of the port
					GENERIC_READ | GENERIC_WRITE,	// Access (read-write) mode
					0,								// Share mode
					NULL,							// Pointer to the security attribute
					OPEN_EXISTING,					// How to open the serial port
					0,								// Port attributes
					NULL							// Handle to port with attribute to copy
					);	

		if(m_hPort!=INVALID_HANDLE_VALUE) {

			//
			// Setup the port
			//
			memset(&dcbCommPort, 0, sizeof(DCB));
			dcbCommPort.BaudRate	= CBR_9600;
			dcbCommPort.ByteSize	= 8;
			dcbCommPort.Parity		= NOPARITY;
			dcbCommPort.StopBits	= ONESTOPBIT;
		//	dcbCommPort.fOutxCtsFlow = TRUE;
		//	dcbCommPort.fRtsControl = RTS_CONTROL_HANDSHAKE;
			SetCommState(m_hPort, &dcbCommPort);

			//
			// Set read and write timeouts to infinite
			//
			cto.ReadIntervalTimeout			= 0;
			cto.ReadTotalTimeoutMultiplier	= 0;
			cto.ReadTotalTimeoutConstant	= 0;
			cto.WriteTotalTimeoutMultiplier = 0;
			cto.WriteTotalTimeoutConstant	= 0;
			SetCommTimeouts(m_hPort, &cto);

			//
			// Initialise the Sned and Receive threads
			//
			fContinue		= TRUE;
			m_hEventWrite	= CreateEvent(NULL, FALSE, FALSE, NULL);
			pThreadRead		= AfxBeginThread(ReadThread, this, THREAD_PRIORITY_NORMAL);
			pThreadWrite	= AfxBeginThread(SendThread, this, THREAD_PRIORITY_NORMAL);
			//
			// Enable buttons that can be pressed
			//
			GetDlgItem(IDC_CLOSE)->EnableWindow(TRUE);
			GetDlgItem(IDC_OPEN)->EnableWindow(FALSE);
			GetDlgItem(IDC_WRITE)->EnableWindow(TRUE);
		}
		else {
			//
			// Failed to open - return false
			//
			DWORD dwErrorCode=GetLastError();
			TCHAR strError[MAX_PATH];
			wsprintf(strError, L"Lasterror returned 0x%0x", dwErrorCode);
			TCHAR* lpBuffer;
			lpBuffer = new TCHAR[512]; //will be deleted in function!
			GetLastErrorText(dwErrorCode, lpBuffer);
			return FALSE;
		}
	}
	return TRUE;
}

//
//	Name: ClosePort
//
//
//	Purpose:	Close the currently opened port
//				Terminate the Send and Receive threads
//
BOOL CVCPTestDlg::ClosePort()
{
	if(m_hPort) {
		CloseHandle(m_hPort);
		m_hPort = NULL;
	}

	//
	// Stop both threads from looping
	//
	fContinue = FALSE;

	//
	// Stop the writing thread by signal
	//
	SetEvent(m_hEventWrite);

	if(WaitForSingleObject(pThreadRead->m_hThread, READ_THREAD_WAIT_TIMEOUT) == WAIT_TIMEOUT) {
		//
		// Didn't exit nicely so kill it
		//
		MessageBox(L"Error RThread");
		TerminateThread(pThreadRead->m_hThread, 0xffffffff);
	}

	//
	// Release this resource
	//
	if(m_hEventWrite) {
		CloseHandle(m_hEventWrite);
	}

	if(WaitForSingleObject(pThreadWrite->m_hThread, WRITE_THREAD_WAIT_TIMEOUT) == WAIT_TIMEOUT) {
		//
		// Didn't exit nicely so kill it
		//
		MessageBox(L"Error WThread");
		TerminateThread(pThreadWrite->m_hThread, 0xffffffff);
	}

	//
	// Enable buttons that can be pressed
	//
	GetDlgItem(IDC_CLOSE)->EnableWindow(FALSE);
	GetDlgItem(IDC_OPEN)->EnableWindow(TRUE);
	GetDlgItem(IDC_WRITE)->EnableWindow(FALSE);
	return TRUE;
}

//
//	Name: SetUpComList
//
//
//	Purpose: Fill iDlgID with our COM (or other named) ports	
//				
//
void CVCPTestDlg::SetUpComList(INT iDlgID)
{
	EnumCOMPorts(iDlgID);
	SendDlgItemMessage (iDlgID, CB_SETCURSEL, 0, 0);
}

//
//	Name: EnumCOMPorts
//
//
//	Purpose: Find all COM ports related to FTDI device	
//				
//
int CVCPTestDlg::EnumCOMPorts(INT iDlgID)
{
	INT i = 0, iRet;
	HKEY hKey, hSubKey;
	TCHAR szName[128];
	TCHAR szPrefix[128];
	DWORD dwType, dwSize;

	//
	// Open the Drivers/Active key to search through all devices
	//
	if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, REG_NAME_ACTIVE, 0, 
		0, &hKey) != ERROR_SUCCESS) {
		return 0;
	}

	while (1) {

		//
		// Read ith key to give us the next key decial number into szName - finish loop if error
		//
		dwSize = sizeof(szName);	// LEAVE THIS HERE - doesn't work if removed and made global
		if(RegEnumKeyEx (hKey, i++, szName, &dwSize, NULL, NULL, NULL, NULL) != ERROR_SUCCESS) {
			break;
		}
		//
		// Open Drivers/Active/XX szName=number key.
		//
		iRet = RegOpenKeyEx(hKey, szName, 0, 0, &hSubKey);
		if(iRet !=  ERROR_SUCCESS) {
			continue;
		}

		//
		// Reuse the szName variable to get the "Name" part (holds the COM etc...) of the Drivers/Active/XX key
		//
		dwSize = sizeof(szName);	// LEAVE THIS HERE - doesn't work if removed and made global
		iRet = RegQueryValueEx(hSubKey, TEXT("Name"), 0, &dwType, (PBYTE)szName, &dwSize);
		if(iRet ==  ERROR_SUCCESS) {

			//
			// szName holds the prefix plus index of the device - so check it
			//
			if(CheckDeviceType(hSubKey)) {
				GetDriverPrefix(szPrefix, sizeof(szPrefix), hSubKey);
				if(CheckNameIndex(szName, szPrefix) != -1) {
					//
					// Its one of our devices so insert it into the list box
					//
					SendDlgItemMessage (iDlgID, CB_INSERTSTRING, -1, (LPARAM)szName);
				}
			}
		}
		RegCloseKey(hSubKey);
	}
	RegCloseKey(hKey);
	return 1;
}

//
//	Name: CheckNameIndex
//
//
//	Purpose: Check out name against the Prefix
//				
//
int CVCPTestDlg::CheckNameIndex(TCHAR * szName, TCHAR * szPrefix)
{
	TCHAR cBuf[6];	// our COM or MOC or whatever name
	int iNum;
	int iRet;
		
	wcsncpy(cBuf, szName, 6);
	if(wcsncmp(cBuf, szPrefix, 3) != 0)
		return -1;

	//
	// Get the index from the device name
	//
	iRet = swscanf(&cBuf[3], L"%d:", &iNum);
	if((iRet == 0) || (iRet == EOF))
		return -1;
	
	return iNum;
}

//
//	Name: CheckDeviceType
//
//
//	Purpose: Check it is an FTDI device
//				
//
BOOL CVCPTestDlg::CheckDeviceType(HKEY hSubKey)
{
	INT rc;
	TCHAR szName[128];
	DWORD dwType, dwSize;

	dwSize = sizeof(szName);

	//
	// Get the "Key" value in Drivers\Active\XX into szName
	//
	rc = RegQueryValueEx(hSubKey, REG_NAME_KEY, 0, &dwType, (PBYTE)szName, &dwSize);
	if(rc !=  ERROR_SUCCESS) {
		return FALSE;
	}

	//
	// Check the "Key" value against what we would expect for and FTDI device
	//
	if(wcslen(szName) >= wcslen(REG_NAME_FTDI_KEY)) {
		if(wcsncmp(szName, REG_NAME_FTDI_KEY, wcslen(REG_NAME_FTDI_KEY)) == 0) {
			return TRUE;
		}
	}

	return FALSE;
}

//
//	Name: GetDriverPrefix
//
//
//	Purpose: Get the "Prefix" from the registry
//				
//
BOOL CVCPTestDlg::GetDriverPrefix(TCHAR * szPrefix, DWORD dwPrefixSize, HKEY hSubKey)
{
	INT rc;
	TCHAR szName[128];
	DWORD dwType, dwSize;
	HKEY hKey = NULL;

	dwSize = sizeof(szName);

	//
	// Get the "Key" value in Drivers\Active\XX into szName
	//
	rc = RegQueryValueEx(hSubKey, REG_NAME_KEY, 0, &dwType, (PBYTE)szName, &dwSize);

	//
	// Open our key name
	//
	if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, szName, 0, 0, &hKey) != ERROR_SUCCESS) {
		return FALSE;
	}

	//
	// Get the "Prefix" key into szPrefix
	//
	rc = RegQueryValueEx(hKey, REG_NAME_PREFIX, 0, &dwType, (PBYTE)szPrefix, &dwPrefixSize);
	if(rc !=  ERROR_SUCCESS) {
		return FALSE;
	}

	if(hKey) {
		RegCloseKey(hKey);
	}

	return TRUE;
}

void CVCPTestDlg::GetLastErrorText(LONG er, TCHAR *text)
{
	LPVOID lpMsgBuf;
	FormatMessage(
		FORMAT_MESSAGE_ALLOCATE_BUFFER |
		FORMAT_MESSAGE_FROM_SYSTEM |
		FORMAT_MESSAGE_IGNORE_INSERTS,
		NULL,
		er,
		0, // Default language
		(LPTSTR) &lpMsgBuf,
		0,
		NULL
	);
	// Process any inserts in lpMsgBuf.
	// ...
	// Display the string.
	wcscpy(text, (LPCTSTR)lpMsgBuf);
	::MessageBox( NULL, (LPCTSTR)lpMsgBuf, L"Error", MB_OK | MB_ICONINFORMATION | MB_TOPMOST | MB_SETFOREGROUND);
	// Free the buffer.
	LocalFree( lpMsgBuf );
}
